<?php

namespace App\Http\Controllers\Api\Users;

use App\Events\SendShipmentEvent;
use App\Http\Controllers\Controller;
use App\Http\Requests\ShipmentRequest;
use App\Http\Resources\CityResource;
use App\Http\Resources\CountryResource;
use App\Http\Resources\ExportShipmentResource;
use App\Http\Resources\ForbiddenItemResource;
use App\Http\Resources\GetDriverResource;
use App\Http\Resources\GovernorateResource;
use App\Http\Resources\ImportShipmentResource;
use App\Http\Resources\InsuranceTypeResource;
use App\Http\Resources\ShipmentResource;
use App\Http\Resources\TruckCategoryResource;
use App\Http\Resources\TruckHeightResource;
use App\Http\Resources\TruckLengthResource;
use App\Http\Resources\TruckLoadCapacityResource;
use App\Http\Resources\TruckModelResource;
use App\Http\Resources\TruckNumberOfAxlesResource;
use App\Http\Resources\TruckResource;
use App\Http\Resources\TruckTypeResource;
use App\Http\Resources\TruckWidthResource;
use App\Jobs\SendShipmentToDrivers;
use App\Models\AppSetting;
use App\Models\Driver;
use App\Models\ForbiddenItem;
use App\Models\FromAddress;
use App\Models\FromAddressDetail;
use App\Models\InsuranceType;
use App\Models\Rate;
use App\Models\Shipment;
use App\Models\ShipmentCity;
use App\Models\ShipmentCountry;
use App\Models\ShipmentDate;
use App\Models\ShipmentDriver;
use App\Models\ShipmentGovernorate;
use App\Models\ToAddress;
use App\Models\ToAddressDetails;
use App\Models\Truck;
use App\Models\TruckCategory;
use App\Models\TruckHeight;
use App\Models\TruckLength;
use App\Models\TruckLoadCapacity;
use App\Models\TruckModel;
use App\Models\TruckNumberOfAxles;
use App\Models\TruckType;
use App\Models\TruckWidth;
use App\Models\User;
use App\Services\External\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Throwable;

class ShipmentController extends Controller
{
    public function get_shipment_data()
    {

        $countries = ShipmentCountry::get();
        $governorates = ShipmentGovernorate::get();
        $cities = ShipmentCity::get();
        $types = TruckType::get();
        $models = TruckModel::get();
        $categories = TruckCategory::get();
        $heights = TruckHeight::get();
        $widths = TruckWidth::get();
        $lengths = TruckLength::get();
        $loads = TruckLoadCapacity::get();
        $axles = TruckNumberOfAxles::get();
        $insurance = InsuranceType::get();


        return apiResponse(200, null, [
            'countries' => CountryResource::collection($countries),
            'governorates' => GovernorateResource::collection($governorates),
            'cities' => CityResource::collection($cities),
            'truck_types' => TruckTypeResource::collection($types),
            'truck_models' => TruckModelResource::collection($models),
            'truck_categories' => TruckCategoryResource::collection($categories),
            'truck_heights' => TruckHeightResource::collection($heights),
            'truck_width' => TruckWidthResource::collection($widths),
            'lengths' => TruckLengthResource::collection($lengths),
            'truck_load_capacity' => TruckLoadCapacityResource::collection($loads),
            'truck_number_of_axles' => TruckNumberOfAxlesResource::collection($axles),
            'insurance_types' => InsuranceTypeResource::collection($insurance),
        ]);
    }

    public function governorates($country_id)
    {
        $governorates = ShipmentGovernorate::where('shipment_country_id', $country_id)->get();
        return apiResponse(200, null, GovernorateResource::collection($governorates));
    }

    public function cities($governorate_id)
    {
        $cities = ShipmentCity::where('shipment_governorate_id', $governorate_id)->get();
        return apiResponse(200, null, CityResource::collection($cities));
    }


    public function get_shipment_details(Request $request)
    {

        $shipment_details = Shipment::where('id', $request->shipment_id)->first();
        $shipment_details['truck_details'] = new TruckResource($shipment_details->truck);

        $shipment_details->from_address = FromAddress::select('latitude', 'longitude')->where('id', $shipment_details->from_address_id)->first();

        $from_address_details = FromAddressDetail::select('address_name', 'street_name', 'description')->where('id', $shipment_details->from_address_detail_id)->first();

        if ($shipment_details->from_address) {
            $shipment_details->from_address = [
                'latitude' => $shipment_details->from_address->latitude,
                'longitude' => $shipment_details->from_address->longitude,
                'address_name' => $from_address_details->address_name ?? "",
                'street_name' => $from_address_details->street_name ?? "",
                'description' => $from_address_details->description ?? "",
            ];
        } else {
            $shipment_details->from_address = [
                'latitude' => "",
                'longitude' => "",
                'address_name' => "",
                'street_name' => "",
                'description' => "",
            ];
        }

        // $shipment_details->from_address->address_name = $from_address_details->address_name;

        // $shipment_details->from_address->street_name = $from_address_details->street_name;

        // $shipment_details->from_address->description = $from_address_details->description;

        $shipment_details->to_address = ToAddress::select('latitude', 'longitude')->where('id', $shipment_details->to_address_id)->first();

        $to_address_details = ToAddressDetails::select('address_name', 'street_name', 'description')->where('id', $shipment_details->to_address_detail_id)->first();

        $shipment_details->to_address->address_name = $to_address_details->address_name;

        $shipment_details->to_address->street_name = $to_address_details->street_name;

        $shipment_details->to_address->description = $to_address_details->description;

        $insurance = InsuranceType::where('id', $shipment_details->insurance_type_id)->first();

        $shipment_details->insurance_type = $insurance->type;

        $shipment_details->insurance_price = $insurance->price;

        $shipment_details->drivers = ShipmentDriver::where('shipment_id', $shipment_details->id)->where('status', 1)->get();

        $all_drivers = array();

        foreach ($shipment_details->drivers as $single_driver) {

            $driver_temp = Driver::select('id', 'name', 'profile_image', 'mobile', 'truck_id')->where('id', $single_driver->driver_id)->first();

            $driver_temp->profile_image = asset('images/') . '/' . $driver_temp->profile_image;


            $truck = Truck::where('id', $driver_temp->truck_id)->first();


            if ($truck) {

                $truck_model = TruckModel::where('id', $truck->model_id)->first();

                if ($truck_model) {
                    $driver_temp->truck_model = $truck_model->model;
                }


                if ($truck->number) {
                    $driver_temp->truck_number = $truck->number;
                    $driver_temp->car_number = $truck->number;
                }


                if ($truck->color) {
                    $driver_temp->truck_color = $truck->color;
                }


            }


            array_push($all_drivers, $driver_temp);
        }

        $shipment_details->drivers = $all_drivers;

        $shipment_details->sender_details = User::select('id', 'name', 'image', 'mobile')->where('id', $shipment_details->sender_id)->first();

        $shipment_details->receiver_details = User::select('id', 'name', 'image', 'mobile')->where('id', $shipment_details->receiver_id)->first();

        $shipment_details->shipment_invoice_picture = asset('images/') . '/' . $shipment_details->shipment_invoice_picture;

        $shipment_details->packing_list_picture = asset('images/') . '/' . $shipment_details->packing_list_picture;

        $shipment_details->sender_details->image = asset('images/') . '/' . $shipment_details->sender_details->image;

        $shipment_details->receiver_details->image = asset('images/') . '/' . $shipment_details->receiver_details->image;

        $insuranceTypePrice = InsuranceType::whereId($shipment_details->insurance_type_id)->first()->price;
        $shipmentPrice = $shipment_details->distance * $shipment_details->kilo_price;
        $taxPrice = $shipmentPrice * (AppSetting::key('tax')->first()->value / 100);
        $insuranceFinalPrice = $shipment_details->final_price * ($insuranceTypePrice / 100);
        $shipment_details['tax_price'] = (number_format($taxPrice, 2));
        $shipment_details['insurance_final_price'] = number_format($insuranceFinalPrice, 2);


        return apiResponse(200, null, $shipment_details);
    }


    public function save_shipment(ShipmentRequest $request)
    {
        $user = auth('user')->user();

        $from_address = FromAddress::create([
            'latitude' => $request->from_latitude,
            'longitude' => $request->from_longitude,
            'user_id' => $user->id,
        ]);

        $from_address_detail = FromAddressDetail::create([
            'country_id' => $request->from_country_id,
            'governorate_id' => $request->from_governorate_id,
            'city_id' => $request->from_city_id,
            'address_name' => $request->from_address_name,
            'street_name' => $request->from_street_name,
            'description' => $request->from_description,
            'from_address_id' => $from_address->id,
        ]);

        $to_address = ToAddress::create([
            'latitude' => $request->to_latitude,
            'longitude' => $request->to_longitude,
            'user_id' => $user->id,
        ]);


        $to_address_detail = ToAddressDetails::create([
            'country_id' => $request->to_country_id,
            'governorate_id' => $request->to_governorate_id,
            'city_id' => $request->to_city_id,
            'address_name' => $request->to_address_name,
            'street_name' => $request->to_street_name,
            'description' => $request->to_description,
            'to_address_id' => $to_address->id,
        ]);
        $distance = $this->haversineGreatCircleDistance($request->from_latitude, $request->from_longitude, $request->to_latitude, $request->to_longitude);

        Storage::disk('local')->put('a7maaa', 'str: ' . json_encode($request->all()));

        $type = TruckType::whereId($request->truck_type_id)->first()->kilo_price;
        $category = TruckCategory::whereId($request->truck_category_id)->first()->kilo_price;
        $length = TruckLength::whereId($request->truck_length_id)->first()->kilo_price;
        $width = TruckWidth::whereId($request->truck_width_id)->first()->kilo_price;
        $height = TruckHeight::whereId($request->truck_height_id)->first()->kilo_price;
        $loadCapacity = TruckLoadCapacity::whereId($request->truck_load_capacity_id)->first()->kilo_price;
        $numberOfAxels = TruckNumberOfAxles::whereId($request->truck_number_of_axles_id)->first()->kilo_price;
        $kiloPrice = ($type + $category + $length + $width + $height + $loadCapacity + $numberOfAxels);
        $truck = Truck::create([
            'type_id' => $request->truck_type_id,
            'height_id' => $request->truck_height_id,
            'width_id' => $request->truck_width_id,
            'length_id' => $request->truck_length_id,
            'load_capacity_id' => $request->truck_load_capacity_id,
            'number_of_axles_id' => $request->truck_number_of_axles_id,
            'catrgory_id' => $request->truck_category_id,
            'kilo' => $kiloPrice

        ]);
        if ($request->shipment_invoice_picture) {

            $shipment_invoice_picture = $request->file('shipment_invoice_picture');
            $shipment_invoice_picture_name = Time() . '-' . $shipment_invoice_picture->getClientOriginalName();
            $file_path = public_path() . '/images';
            $shipment_invoice_picture->move($file_path, $shipment_invoice_picture_name);
        }
        if ($request->packing_list_picture) {

            $packing_list_picture = $request->file('packing_list_picture');
            $packing_list_picture_name = Time() . '-' . $packing_list_picture->getClientOriginalName();
            $file_path = public_path() . '/images';
            $packing_list_picture->move($file_path, $packing_list_picture_name);
        }

        $receiver = User::where('mobile', $request->receiver_mobile)->first();

        if (!$receiver) {
            return apiResponse(404, 'user not found', null);
        }

        $ship = Shipment::create([
            'sender_id' => $user->id,
            'from_address_id' => $from_address->id,
            'from_address_detail_id' => $from_address_detail->id,
            'to_address_id' => $to_address->id,
            'to_address_detail_id' => $to_address_detail->id,
            'receiver_id' => $receiver->id,
            'type' => $request->shipment_type,
            'weight' => $request->shipment_weight,
            'description' => $request->shipment_description,
            'truck_id' => $truck->id,
            'insurance_type_id' => $request->insurance_type_id,
            'shipment_invoice_picture' => $shipment_invoice_picture_name ?? null,
            'packing_list_picture' => $packing_list_picture_name ?? null,
            'packing_date' => $request->packing_date,
            'packing_time' => $request->packing_time,
            'number_of_trucks' => $request->number_of_trucks,
            'distance' => $distance,
            'kilo_price' => $kiloPrice,
            'status' => 0,
        ]);

        $shipment = Shipment::where('id', $ship->id)->first();

        $insuranceTypePrice = InsuranceType::whereId($request->insurance_type_id)->first()->price;
        $shipmentPrice = $distance * $kiloPrice;
        $appPrice = $shipmentPrice * (AppSetting::key('app_percentage')->first()->value / 100);
        $taxPrice = $shipmentPrice * (AppSetting::key('tax')->first()->value / 100);
        $finalPrice = (int) ($shipmentPrice + ($shipmentPrice * $insuranceTypePrice / 100) + $appPrice + $taxPrice);
        //dd(['insurance'=>$shipmentPrice * $insuranceTypePrice / 100,'distance'=>$distance,'kilo'=>$kiloPrice,'price'=>$shipmentPrice,'app_price'=>$appPrice,'tax_price'=>$taxPrice,'final_price'=>$finalPrice]);
        $shipment['kilo_price'] = $finalPrice;
        $shipment['user_limit_price_percentage'] = AppSetting::key('user_limit_price_percentage')->first()->value;


        return apiResponse(200, null, $shipment);
    }

    public function haversineGreatCircleDistance(
        float $latitudeFrom,
        float $longitudeFrom,
        float $latitudeTo,
        float $longitudeTo,
        float $earthRadius = 6371
    ): float {
        // convert from degrees to radians
        $latFrom = deg2rad($latitudeFrom);
        $lonFrom = deg2rad($longitudeFrom);
        $latTo = deg2rad($latitudeTo);
        $lonTo = deg2rad($longitudeTo);

        $latDelta = $latTo - $latFrom;
        $lonDelta = $lonTo - $lonFrom;

        $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) +
            cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
        return $angle * $earthRadius;
    }

    public function custom_price(Request $request)
    {
        $shipment = Shipment::findOrFail($request->shipment_id);

        $drivers = Driver::where('stop_receiving_shipment_requests', 1)->select('id')->get();
        $appPrice = $request->price * (AppSetting::key('app_percentage')->first()->value / 100);
        $taxPrice = $request->price * (AppSetting::key('tax')->first()->value / 100);
        foreach ($drivers as $driver) {
            $shipment->drivers()->attach([
                $driver->id => ['price' => $request->price, 'app_price' => $appPrice, 'tax' => $taxPrice]
            ]);
            SendShipmentToDrivers::dispatch($shipment, $driver->id);


        }

        $shipment->update(['user_price' => $request->price]);

        return apiResponse(200, 'user price is updated', new ShipmentResource($shipment));
    }

    public function current_export_shipments()
    {
        $user = auth('user')->user();

        $shipments = $user->sender_shipments()->where('status', '!=', 4)
            ->orderBy('shipments.created_at', 'desc')->get();

        foreach ($shipments as $shipment) {
            $count = count($shipment->drivers);
            $shipment['count'] = $count;
        }

        return apiResponse(200, null, ImportShipmentResource::collection($shipments));
    }

    public function current_import_shipments()
    {


        $user = auth('user')->user();
        $shipments = Shipment::with([
            "drivers" => function ($q) {
                $q->where('drivers.status', 1);
            }
        ])->where('sender_id', $user->id)->where('shipments.status', '!=', 4)
            ->orderBy('shipments.created_at', 'desc')
            ->get();


        foreach ($shipments as $shipment) {
            $count = count($shipment->drivers);
            $shipment->count = $count;
        }
        return apiResponse(200, null, ExportShipmentResource::collection($shipments));
    }

    public function previous_export_shipments()
    {
        $user = auth('user')->user();
        $shipments = $user->sender_shipments()->where('status', 4)
            ->orderBy('shipments.created_at', 'desc')->get();
        return apiResponse(200, null, ImportShipmentResource::collection($shipments));
    }


    public function previous_import_shipments()
    {

        $user = auth('user')->user();


        $shipments = Shipment::with([
            "drivers" => function ($q) {
                $q->where('status', 1);
            }
        ])->where('receiver_id', $user->id)->where('status', 4)->get();


        foreach ($shipments as $shipment) {
            $count = count($shipment->drivers);
            $shipment['count'] = $count;
        }
        return apiResponse(200, null, ExportShipmentResource::collection($shipments));
    }

    public function get_drivers(Request $request)
    {
        $drivers = Driver::whereIn('id', ShipmentDriver::where('shipment_id', $request->shipment_id)->where('status', 1)->pluck('driver_id'))->get();


        $shipment = Shipment::where('id', $request->shipment_id)->first();

        $drivers_Count = count($drivers);

        $number_of_trucks = $shipment->number_of_trucks;

        foreach ($drivers as $driver) {
            $total = 0;
            $user_rates = Rate::where('driver_id', $driver->id)->get();
            if (count($user_rates) > 0) {
                foreach ($user_rates as $rate) {
                    $total += $rate->rate;
                }
                $total_rate = $total / count($user_rates);
                $driver->rates = $total_rate;
            } else {
                $driver->rates = 0;
            }
            $driver['drivers_Count'] = $drivers_Count;
            $driver['number_of_trucks'] = $number_of_trucks;
            $shipmentDriver = ShipmentDriver::where(['driver_id' => $driver->id, 'shipment_id' => $request->shipment_id])->first();
            $driver['price'] = $shipmentDriver->price;
        }

        return apiResponse(200, null, GetDriverResource::collection($drivers));
    }

    public function select_driver(Request $request)
    {
        $shipment = Shipment::where('id', $request->shipment_id)->first();
        $shipment->status = 1;
        $shipment->save();

        return apiResponse(200, null, new ShipmentResource($shipment));
    }

    public function accept_driver(Request $request)
    {
        $shipment = Shipment::whereId($request->shipment_id)->first();

        $shipmentDriver = ShipmentDriver::where(['driver_id' => $request->driver_id, 'shipment_id' => $request->shipment_id])->where('status', 1)->first();
        $shipmentDriver->update(['status' => 2]);
        $shipment->update(['final_price' => $shipmentDriver->price]);
        Notification::notify('drivers', 'shipments', [$request->driver_id], $shipment->id, 'تم الموافقة علي السعر', 'تم الموافقة علي السعر');
        if (count($shipment->selectedDriver) == $shipment->number_of_trucks) {
            $shipment->unAcceptedDriver()
                ->wherePivot('status', 0)
                ->detach();
        }
        broadcast(new SendShipmentEvent($shipment, $request->driver_id, 'accepted-driver'));

        return apiResponse(200, 'driver price is accepted', null);
    }

    public function cancel_shipment($shipment)
    {
        $user = auth('user')->user();
        $ship = Shipment::findOrFail($shipment);

        if ($ship->sender_id == $user->id) {
            if ($ship->status == 0 || $ship->status == 1 || $ship->status == 5) {
                Notification::notify('users', 'shipments', [$user->id], $ship->id, 'new-shipment', 'new-shipment');
                $ship->delete();
                $drivers = ShipmentDriver::where('shipment_id', $shipment)->where('status', 1)->get();
                foreach ($drivers as $driver) {
                    broadcast(new SendShipmentEvent($shipment, $driver->driver_id, 'cancel-shipment'));
                }

                return apiResponse(200, 'shipment is deleted', null);
            }
            return apiResponse(401, 'you cant cancel shipment', null);
        }

        return apiResponse(404, 'shipment not found', null);
    }

    public function search(Request $request)
    {
        try {
            $query = Shipment::query();

            if ($request->keyword) {

                $keyword = $request->keyword;

                $shipments = $query->where('user_price', 'like', '%' . $keyword . '%')
                    ->orWhere('description', 'like', '%' . $keyword . '%')
                    ->orWhere('type', 'like', '%' . $keyword . '%')
                    ->orWhereHas('user', function ($item) use ($keyword) {
                        $item->where('name', 'like', '%' . $keyword . '%');
                    })->get();

                if (count($shipments)) {
                    return apiResponse(200, null, ShipmentResource::collection($shipments));
                }
                return apiResponse(404, 'no results', null);
            }
            return apiResponse(404, 'keyword not found', null);

        } catch (Throwable $th) {
            return apiResponse($th->getCode(), $th->getMessage(), null);
        }
    }

    public function forbidden_items()
    {
        try {
            $items = ForbiddenItem::get();
            return apiResponse(200, null, ForbiddenItemResource::collection($items));

        } catch (Throwable $th) {
            return apiResponse(404, $th->getMessage(), null);
        }
    }

    public function remove_driver(Request $request)
    {
        $shipment = ShipmentDriver::where('shipment_id', $request->shipment_id)->where('driver_id', $request->driver_id)->first();
        $shipment->delete();
        return apiResponse(200, null, null);
    }
}
